const mongoose = require('mongoose');

// Define the schema for a Payment transaction
const paymentSchema = new mongoose.Schema({
  // Reference to the Labourer who made the payment
  labourerId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Labor', // Assumes you have a 'Labour' model
    required: true,
  },
  // Reference to the WorkOrder associated with this payment
  workOrderId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'WorkOrder', // Assumes you have a 'WorkOrder' model
    required: true,
  },
  // The amount of money involved in the transaction
  amount: {
    type: Number,
    required: true,
    min: 0, // Ensure amount is non-negative
  },
  // The currency of the transaction (e.g., PKR for Pakistani Rupees)
  currency: {
    type: String,
    required: true,
    default: 'PKR', // Default to Pakistani Rupees as per request
  },
  // Type of transaction from the labourer's perspective:
  // 'debit' means money is leaving the labourer's account (paying admin)
  // 'credit' would be if money is coming into their account (e.g., refund)
  transactionType: {
    type: String,
    enum: ['debit', 'credit'],
    required: true,
  },
  // Status of the payment (e.g., if integrated with a real payment gateway)
  // For this implementation, we'll assume 'completed' for simplicity as it's a simulated internal transaction.
  status: {
    type: String,
    enum: ['pending', 'completed', 'failed'],
    default: 'completed',
  },
  // A brief description of the payment
  description: {
    type: String,
    default: 'Work order acceptance fee',
  },
}, {
  // Add timestamps to automatically record createdAt and updatedAt fields
  timestamps: true,
});

// Create and export the Payment model
module.exports = mongoose.model('Payment', paymentSchema);
